/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.sax;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import de.avm.android.tr064.Tr064Capabilities;

/*
 * SAX-Handler for FRITZ!Box WLANConfiguration1 description
 */
public class SAXWlanConfScpdHandler extends SAXScpdHandler
{
	private static final int SPECIFIC_INDEX = 10;
	private static final int HYBRID_INDEX   = 11;
	private static final int CONF_EXT_INDEX = 13;
	private static final int WPS_INDEX = 14;

	private static final String STATEVAR_PATH = "/scpd/serviceStateTable/stateVariable";
	private static final String STATEVAR_RANGE_MIN_PATH = "/scpd/serviceStateTable/stateVariable/allowedValueRange/minimum";
	private static final String STATEVAR_RANGE_MAX_PATH = "/scpd/serviceStateTable/stateVariable/allowedValueRange/maximum";
	private static final String STATEVAR_SIGNAL_NAME = "X_AVM-DE_SignalStrength";

	private boolean mInVarName = false;
	private String mCurrentName = "";
	private boolean mInSignal = false;
	private String mCurrentMin = "";
	private String mCurrentMax = "";
	
	private String mSignalMin = "";
	private String mSignalMax = "";
	
	public boolean isCapabilitiyOfInterface(Tr064Capabilities cap)
	{
		return cap.has(Tr064Capabilities.Capability.WLAN_CONF) ||
				cap.has(Tr064Capabilities.Capability.WLAN_CONF_SPECIFIC) ||
				cap.has(Tr064Capabilities.Capability.WLAN_CONF_HYBRID) ||
				cap.has(Tr064Capabilities.Capability.WLAN_CONF_EXT) ||
				cap.has(Tr064Capabilities.Capability.WLAN_CONF_WPS);
	}
	
	public String getSignalRangeMin()
	{
		return mSignalMin;
	}
	
	public String getSignalRangeMax()
	{
		return mSignalMax;
	}
	
	/**
	 * Instantiates a new SAX FRITZ!Box WLANConfiguration1 description handler.
	 */
	public SAXWlanConfScpdHandler()
	{
		initActionsArray(new String[]
  		{
			// Capability.WLAN_CONF
			"SetBeaconType",
			"SetSecurityKeys",
			"SetSSID",
			"SetEnable",
			"GetInfo",
			"GetSecurityKeys",
			"GetSSID",
			"GetBSSID",
			"GetTotalAssociations",
			"GetGenericAssociatedDeviceInfo",
			
			// Capability.WLAN_CONF_SPECIFIC
			"GetSpecificAssociatedDeviceInfo",
			
			// Capability.WLAN_CONF_HYBRID
			"X_AVM-DE_SetWLANHybridMode",
			"X_AVM-DE_GetWLANHybridMode",
			
			// Capability.WLAN_CONF_EXT
			"X_AVM-DE_GetWLANExtInfo",
			
			// Capability.WLAN_CONF_WPS
			"X_AVM-DE_GetWPSInfo",
			"X_AVM-DE_SetWPSConfig"
  		});
	}

	@Override
	public Tr064Capabilities getCapabilities()
	{
		boolean available = true;
		for(int ii = 0; available && (ii < SPECIFIC_INDEX); ii++)
			if (!mActions[ii].mIsAvailable)
				available = false;

		if (available)
		{
			Tr064Capabilities result = new Tr064Capabilities(
					Tr064Capabilities.Capability.WLAN_CONF);

			// specific
			if (mActions[SPECIFIC_INDEX].mIsAvailable &&
					(mActions[SPECIFIC_INDEX].mArguments != null) &&
					mActions[SPECIFIC_INDEX].mArguments.contains("NewX_AVM-DE_Speed") &&
					mActions[SPECIFIC_INDEX].mArguments.contains("NewX_AVM-DE_SignalStrength"))
				result.add(Tr064Capabilities.Capability.WLAN_CONF_SPECIFIC);

			// hybrid
			available = true;
			for(int ii = HYBRID_INDEX; available && (ii < CONF_EXT_INDEX); ii++)
				if (!mActions[ii].mIsAvailable)
					available = false;
			if (available)
				result.add(Tr064Capabilities.Capability.WLAN_CONF_HYBRID);
			
			// wlan conf ext
			if (mActions[CONF_EXT_INDEX].mIsAvailable)
				result.add(Tr064Capabilities.Capability.WLAN_CONF_EXT);
			
			// wlan WPS
			available = true;
			for(int ii = WPS_INDEX; available && (ii < mActions.length); ii++)
				if (!mActions[ii].mIsAvailable)
					available = false;
			if (available)
				result.add(Tr064Capabilities.Capability.WLAN_CONF_WPS);
			
			return result;
		}
		
		return Tr064Capabilities.EMPTY;
	}
	
	@Override
	public void startElement(String namespaceURI, String localName,
			String qName, Attributes atts) throws SAXException
	{
		super.startElement(namespaceURI, localName, qName, atts);
		
		if (getCurrentPath().equalsIgnoreCase(STATEVAR_PATH))
		{
			mInSignal = false;
			mCurrentMin = "";
			mCurrentMax = "";
		}
		else if (getCurrentPath().equalsIgnoreCase(STATEVAR_PATH + "/name"))
		{
			mInVarName = true;
			mCurrentName = "";
		}
	}

	@Override
	public void endElement(String namespaceURI, String localName, String qName)
			throws SAXException
	{
		if (mInVarName)
		{
			mInVarName = false;
			if (mCurrentName.equalsIgnoreCase(STATEVAR_SIGNAL_NAME))
				mInSignal = true;
		}
		else if (mInSignal && getCurrentPath().equalsIgnoreCase(STATEVAR_PATH))
		{
			mInSignal = false;
			if (mCurrentMin.length() > 0) mSignalMin = mCurrentMin;
			if (mCurrentMax.length() > 0) mSignalMax = mCurrentMax;
		}
		
		super.endElement(namespaceURI, localName, qName);
	}

	@Override
	public void characters(char ch[], int start, int length)
	{
		super.characters(ch, start, length);
		
		String str = new String(ch, start, length);
		if (str.length() > 0)
		{
			if (mInVarName)
			{
				mCurrentName += str;
			}
			else if (getCurrentPath().equalsIgnoreCase(STATEVAR_RANGE_MIN_PATH))
			{
				mCurrentMin += str;
			}
			else if (getCurrentPath().equalsIgnoreCase(STATEVAR_RANGE_MAX_PATH))
			{
				mCurrentMax += str;
			}
		}
	}
}
